package fit.util;

import java.util.Arrays;
import fit.framework.Theory;

/**
 * Theory implementation based on arrays.
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public class ArraysTheory implements Theory {

    private final String name_;
    private final double[] xs_;
    private final double[] ys_;
    private final int npoint_;

    /**
     * Constructor.  The X and Y arrays must be the same length, and the
     * X values must be sorted (strictly monotonic increasing, no NaNs).
     *
     * @param   name  theory name
     * @param   xs    array of X values
     * @param   ys    array of Y values
     */
    public ArraysTheory( String name, double[] xs, double[] ys ) {
        if ( xs.length != ys.length ) {
            throw new IllegalArgumentException( "Array length mismatch" );
        }
        for ( int i = 1; i < xs.length; i++ ) {
            if ( ! ( xs[ i ] > xs[ i - 1 ] ) ) {
                throw new IllegalArgumentException( "Non-increasing X values" );
            }
        }
        name_ = name;
        xs_ = xs;
        ys_ = ys;
        npoint_ = xs.length;
    }

    public String getName() {
        return name_;
    }

    public int getCount() {
        return npoint_;
    }

    public double getX( int i ) {
        return xs_[ i ];
    }

    public double getY( int i ) {
        return ys_[ i ];
    }

    public int getFloorIndex( double x ) {
        int pos = Arrays.binarySearch( xs_, x );

        /* If there's an exact hit, return the index. */
        if ( pos >= 0 ) {
            return pos;
        }

        /* Otherwise we have an insertion point for the X value.
         * The floor is the index before it. */
        else {
            int insertPoint = - pos - 1;
            return insertPoint - 1;
        }
    }
}
