package fit.util;

import fit.framework.MetadataItem;
import fit.framework.MetadataTable;
import fit.framework.Theory;
import fit.framework.TheorySet;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

/**
 * TheorySet implementation based on an array.
 *
 * @author   Mark Taylor
 * @since    8 Nov 2006
 */
public class ArraysTheorySet implements TheorySet {

    private final Theory[] theories_;
    private final MetadataTable metaTable_;
    private final MetadataItem metaX_;
    private final MetadataItem metaY_;
    private final Map indexMap_;

    private static final Logger logger_ = Logger.getLogger( "fit.util" );

    /**
     * Constructor.
     *
     * @param   theories  array of theories
     * @param   metaTable  metadata table, or null
     */
    public ArraysTheorySet( Theory[] theories, MetadataItem metaX,
                            MetadataItem metaY, MetadataTable metaTable ) {
        theories_ = theories;
        metaX_ = metaX;
        metaY_ = metaY;
        metaTable_ = metaTable;
        indexMap_ = new HashMap();
        for ( int ith = 0; ith < theories.length; ith++ ) {
            registerTheory( theories[ ith ], ith );
        }
        if ( indexMap_.size() != theories.length ) {
            logger_.warning( "Some duplicated theories" 
                           + " - getIndex() may behave strangely" );
        }
    }

    public int getTheoryCount() {
        return theories_.length;
    }

    public Theory getTheory( int ith ) {
        return theories_[ ith ];
    }

    public int getIndex( Theory theory ) {
        return recursiveGetIndex( theory );
    }

    /**
     * Returns the index of a theory relative to this set.  If it doesn't
     * have one, tries to find the index of a theory that it wraps.
     *
     * @param  theory  theory
     * @return  index or -1
     */
    private int recursiveGetIndex( Theory theory ) {
        if ( indexMap_.containsKey( theory ) ) {
            return ((Integer) indexMap_.get( theory )).intValue();
        }
        else if ( theory instanceof WrapperTheory ) {
            return recursiveGetIndex( ((WrapperTheory) theory).getBase() );
        }
        else {
            return -1;
        }
    }

    public MetadataItem getMetadataX() {
        return metaX_;
    }

    public MetadataItem getMetadataY() {
        return metaY_;
    }

    public MetadataTable getMetadataTable() {
        return metaTable_;
    }

    /**
     * Indicates that a given theory is associated with a given index
     * of this set.  This can be used to make {@link #getIndex} return
     * useful values for theories other than those actually in this set.
     *
     * @param   theory  theory
     * @param   index  index associated with <code>theory</code>
     */
    public final void registerTheory( Theory theory, int index ) {
        if ( index >= -1 && index < getTheoryCount() ) {
            indexMap_.put( theory, new Integer( index ) );
        }
        else {
            throw new IllegalArgumentException();
        }
    }
}
