package fit.util;

import fit.framework.Interpolator;

/**
 * Smoother which uses a uniform smoothing kernel.
 *
 * @author   Mark Taylor
 * @since    23 Oct 2006
 */
public class SquareSmoother extends Smoother {

    private final double err_;

    /**
     * Constructor.
     *
     * @param  width  full width (error x 2) of the square kernel
     */
    public SquareSmoother( double width ) {
        if ( width <= 0 ) {
            throw new IllegalArgumentException( "Non-positive width " + width );
        }
        err_ = width * 0.5;
    }

    public double[] getBounds( double x ) {
        return new double[] { x - err_, x + err_, };
    }

    public double getWeight( double xMean, double xPoint ) {
        boolean within = Math.abs( xPoint - xMean ) <= err_;
        assert within : "getWeight called outside of defined range";
        return within ? 1.0 : 0.0;
    }

    public String toString() {
        return "SquareSmoother(width=" + ( err_ * 2 ) + ")";
    }

    /**
     * Returns an InterpolatorFactory which dispenses
     * <code>SquareSmoother</code>s.
     */
    public static InterpolatorFactory getFactory() {
        return new InterpolatorFactory() {
            public Interpolator getInterpolator( double x, double xerr ) {
                return new SquareSmoother( xerr * 2 );
            }
        };
    }
}
