package fit.util;

import fit.framework.MetadataTable;
import java.io.IOException;
import java.util.logging.Logger;
import uk.ac.starlink.table.ColumnInfo;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.table.Tables;

/**
 * MetadataTable implementation based on a <code>StarTable</code>.
 *
 * @author   Mark Taylor
 * @since    7 Nov 2006
 */
public class StarMetadataTable implements MetadataTable {

    private final StarTable starTable_;
    private final ColumnInfo[] colInfos_;

    private static final Logger logger_ = Logger.getLogger( "fit.util" );

    /**
     * Constructor
     *
     * @param  starTable   StarTable supplying data and metadata
     */
    public StarMetadataTable( StarTable starTable ) throws IOException {
        starTable_ = Tables.randomTable( starTable );
        colInfos_ = Tables.getColumnInfos( starTable_ );
    }

    /**
     * Returns the StarTable underlying this object.
     * The return value may be a randomised copy of the original.
     *
     * @return   star table
     */
    public StarTable getStarTable() {
        return starTable_;
    }

    public int getColumnCount() {
        return starTable_.getColumnCount();
    }

    public int getRowCount() {
        return Tables.checkedLongToInt( starTable_.getRowCount() );
    }

    public String getColumnName( int icol ) {
        return colInfos_[ icol ].getName();
    }

    public Class getColumnClass( int icol ) {
        return colInfos_[ icol ].getContentClass();
    }

    public Object getValueAt( int irow, int icol ) {
        try {
            return starTable_.getCell( (long) irow, icol );
        }
        catch ( IOException e ) {
            logger_.warning( "Error reading cell (" 
                           + irow + ", " + icol + ")" );
            return null;
        }
    }
}
