package fit.gui;

import fit.framework.Theory;
import fit.framework.TheorySet;
import java.util.Arrays;
import org.jfree.data.DomainOrder;
import org.jfree.data.general.AbstractSeriesDataset;
import org.jfree.data.xy.XYDataset;
import uk.ac.starlink.util.IntList;

/**
 * XYDataset implementation representing a TheorySet.
 * This is an adaptor.
 * The {@link #setSelected} method allows selection of which theories
 * from the TheorySet are represented in this Dataset (default is all).
 *
 * @author   Mark Taylor
 * @since    22 Jan 2007
 */
public class TheorySetDataset extends AbstractSeriesDataset 
                              implements XYDataset {

    private final TheorySet tSet_;
    private int[] included_;

    /**
     * Constructor.
     *
     * @param   tset  theory set
     */
    public TheorySetDataset( TheorySet tset ) {
        tSet_ = tset;
        boolean[] selected = new boolean[ tset.getTheoryCount() ];
        Arrays.fill( selected, true );
        setSelected( selected );
    }

    /**
     * Returns the TheorySet for which this Dataset is an adaptor.
     *
     * @return  theory set
     */
    public TheorySet getTheorySet() {
        return tSet_;
    }

    /**
     * Sets the theories in the theory set which will be represented by this
     * dataset.  Only those theories with a true value in the corresponding
     * element of the <code>selected</code> array will appear.
     * Note that the <code>selected</code> array is used to configure state
     * and then discarded; subsequent modifications of it will not affect
     * behaviour of this object.
     *
     * @param   selected  map of which theories to represent
     */
    public void setSelected( boolean[] selected ) {
        IntList incList = new IntList();
        for ( int i = 0; i < selected.length; i++ ) {
            if ( selected[ i ] ) {
                incList.add( i );
            }
        }
        included_ = incList.toIntArray();
        fireDatasetChanged();
    }

    private Theory getTheory( int is ) {
        return tSet_.getTheory( included_[ is ] );
    }

    public DomainOrder getDomainOrder() {
        return DomainOrder.ASCENDING;
    }

    public int getSeriesCount() {
        return included_.length;
    }

    public Comparable getSeriesKey( int is ) {
        return getTheory( is ).getName();
    }

    public int getItemCount( int is ) {
        return getTheory( is ).getCount();
    }

    public double getXValue( int is, int ip ) {
        return getTheory( is ).getX( ip );
    }

    public double getYValue( int is, int ip ) {
        return getTheory( is ).getY( ip );
    }

    public Number getX( int is, int ip ) {
        return new Double( getXValue( is, ip ) );
    }

    public Number getY( int is, int ip ) {
        return new Double( getYValue( is, ip ) );
    }
}
