package fit.task;

import fit.formats.ModelTableTheoryWriter;
import fit.framework.MetadataItem;
import fit.framework.Theory;
import fit.framework.TheorySet;
import fit.util.ArraysTheorySet;
import fit.util.TheoryFactory;
import fit.util.WrapperTheorySet;
import gnu.jel.CompilationException;
import gnu.jel.CompiledExpression;
import gnu.jel.Evaluator;
import gnu.jel.Library;
import java.io.IOException;
import uk.ac.starlink.fits.FitsTableWriter;
import uk.ac.starlink.table.DefaultValueInfo;
import uk.ac.starlink.table.StarTableWriter;
import uk.ac.starlink.task.ChoiceParameter;
import uk.ac.starlink.task.Environment;
import uk.ac.starlink.task.Executable;
import uk.ac.starlink.task.OutputStreamParameter;
import uk.ac.starlink.task.Parameter;
import uk.ac.starlink.task.ParameterValueException;
import uk.ac.starlink.task.SingleTaskInvoker;
import uk.ac.starlink.task.Task;
import uk.ac.starlink.task.TaskException;
import uk.ac.starlink.task.TerminalEnvironment;
import uk.ac.starlink.ttools.JELUtils;
import uk.ac.starlink.util.Destination;
import uk.ac.starlink.votable.DataFormat;
import uk.ac.starlink.votable.VOTableWriter;

/**
 * Task which copies a model data file from one format to another, 
 * optionally transforming the X/Y (wavelength/flux) values at the same time.
 *
 * @author   Mark Taylor
 * @since    19 Jan 2007
 */
public class CopyTheorySetTask implements Task {

    private final TheorySetParameter itsetParam_;
    private final OutputStreamParameter otsetParam_;
    private final ChoiceParameter ofmtParam_;
    private final Parameter xParam_;
    private final Parameter yParam_;
    private final Parameter xunitParam_;
    private final Parameter yunitParam_;
    private final Parameter xnameParam_;
    private final Parameter ynameParam_;

    /**
     * Constructor.
     */
    public CopyTheorySetTask() {
        itsetParam_ = new TheorySetParameter( "in" );
        itsetParam_.setPrompt( "Location of input model data" );
        itsetParam_.getReaderParameter().setName( "ifmt" );
        itsetParam_.setDescription( new String[] {
            "<p>Input model data files.",
            "The format is given by the",
            "<code>" + itsetParam_.getReaderParameter() + "</code> parameter.",
            "Multiple model files (in the same format) may be specified",
            "by giving this parameter multiple times.",
            "</p>",
        } );

        otsetParam_ = new OutputStreamParameter( "out" );
        otsetParam_.setPrompt( "Location of output model data" );
        otsetParam_.setDescription( new String[] {
            "<p>Gives the name of the output file to which the model data",
            "will be written in <code>ymodel</code> format.",
            "</p>",
        } );

        ofmtParam_ = new ChoiceParameter( "ofmt" );
        ofmtParam_.addOption( new FitsTableWriter(), "ymodel-fits" );
        ofmtParam_.addOption( new VOTableWriter(), "ymodel-votable" );
        ofmtParam_.addOption( new VOTableWriter( DataFormat.BINARY, true ),
                              "ymodel-votable-binary" );
        ofmtParam_.setPrompt( "Format for output model data" );
        ofmtParam_.setDefault( "ymodel-votable" );
        ofmtParam_.setDescription( new String[] {
            "<p>Gives the format in which the output will be written.",
            "The basic format is <code>ymodel</code>, but it is possible",
            "to choose which variant (FITS, VOTable etc) of that format",
            "is used by specifying this parameter.  It doesn't matter much",
            "which is used though there may be performance implications.",
            "</p>",
        } );

        xParam_ = new Parameter( "x" );
        xParam_.setUsage( "<expr>" );
        xParam_.setPrompt( "Formula for output X value (in x and y)" );
        xParam_.setDefault( "x" );
        xParam_.setDescription( new String[] {
            "<p>Formula for the X (wavelength) values in the output",
            "model file in terms of the X values in the input model file.",
            "The default value of \"<code>x</code>\" simply makes the one",
            "equal to the other, however it is possible to use algebraic",
            "expressions here in terms of <code>x</code> and <code>y</code>",
            "(the Y values from the input model file).",
            "Normal arithmetic operators as well as some special functions",
            "may be used - see <ref id='jel'/>.",
            "</p>",
        } );

        yParam_ = new Parameter( "y" );
        yParam_.setUsage( "<expr>" );
        yParam_.setPrompt( "Formula for output Y value (in x and y)" );
        yParam_.setDefault( "y" );
        yParam_.setDescription( new String[] {
            "<p>Formula for the Y (flux) values in the output",
            "model file in terms of the Y values in the input model file.",
            "The default value of \"<code>y</code>\" simply makes the one",
            "equal to the other, however it is possible to use algebraic",
            "expressions here in terms of <code>y</code> and <code>x</code>",
            "(the X values from the input model file).",
            "Normal arithmetic operators as well as some special functions",
            "such as <code>abToJansky()</code> may be used -",
            "see <ref id='jel'/>.",
            "</p>",
        } );

        xnameParam_ = new Parameter( "xname" );
        xnameParam_.setNullPermitted( true );
        xnameParam_.setPrompt( "New name for output X values" );
        xnameParam_.setDescription( new String[] {
            "<p>Gives a name for the output X axis, such as \"Wavelength\".",
            "This does not affect processing, but may be used to annotate",
            "the displayed or output results, so it is a good idea to supply",
            "a value if known to reduce confusion.",
            "</p>",
        } );

        xunitParam_ = new Parameter( "xunit" );
        xunitParam_.setNullPermitted( true );
        xunitParam_.setPrompt( "New units for output X values" );
        xunitParam_.setDescription( new String[] {
            "<p>Gives a unit for the output X axis, such as \"Angstrom\".",
            "This does not affect processing, but may be used to annotate",
            "the displayed or output results, so it is a good idea to supply",
            "a value if known to reduce confusion.",
            "</p>",
        } );

        ynameParam_ = new Parameter( "yname" );
        ynameParam_.setNullPermitted( true );
        ynameParam_.setPrompt( "New name for output Y values" );
        ynameParam_.setDescription( new String[] {
            "<p>Gives a name for the output Y axis, such as \"Flux\".",
            "This does not affect processing, but may be used to annotate",
            "the displayed or output results, so it is a good idea to supply",
            "a value if known to reduce confusion.",
            "</p>",
        } );

        yunitParam_ = new Parameter( "yunit" );
        yunitParam_.setNullPermitted( true );
        yunitParam_.setPrompt( "New units for output Y values" );
        yunitParam_.setDescription( new String[] {
            "<p>Gives a unit for the output Y axis, such as \"Jansky\".",
            "This does not affect processing, but may be used to annotate",
            "the displayed or output results, so it is a good idea to supply",
            "a value if known to reduce confusion.",
            "</p>",
        } );
    }

    public String getPurpose() {
        return "Converts and/or transforms spectrum model files";
    }

    public Parameter[] getParameters() {
        return new Parameter[] {
            itsetParam_,
            itsetParam_.getReaderParameter(),
            otsetParam_,
            ofmtParam_,
            xParam_,
            yParam_,
            xnameParam_,
            xunitParam_,
            ynameParam_,
            yunitParam_,
        };
    }

    public Executable createExecutable( Environment env ) throws TaskException {
        TheorySet itset = itsetParam_.theorySetValue( env );
        final StarTableWriter twriter =
            (StarTableWriter) ofmtParam_.objectValue( env );
        final Destination dest = otsetParam_.destinationValue( env );
        String xEx = xParam_.stringValue( env );
        String yEx = yParam_.stringValue( env );
        final CompiledExpression xCex;
        final CompiledExpression yCex;
        if ( "x".equalsIgnoreCase( xEx ) && "y".equalsIgnoreCase( yEx ) ) {
            xCex = null;
            yCex = null;
        }
        else {
            Library lib = getLibrary();
            try {
                xCex = Evaluator.compile( xEx, lib, double.class );
            }
            catch ( CompilationException e ) {
                throw new ParameterValueException( xParam_, e.getMessage(), e );
            }
            try {
                yCex = Evaluator.compile( yEx, lib, double.class );
            }
            catch ( CompilationException e ) {
                throw new ParameterValueException( yParam_, e.getMessage(), e );
            }
        }
        String xname = xnameParam_.stringValue( env );
        String xunit = xunitParam_.stringValue( env );
        String yname = ynameParam_.stringValue( env );
        String yunit = yunitParam_.stringValue( env );
        final TheorySet tset;
        if ( ! isBlank( xunit ) || ! isBlank( xname ) ||
             ! isBlank( yunit ) || ! isBlank( yname ) ) {
            xname = isBlank( xname ) ? itset.getMetadataX().getName() : xname;
            yname = isBlank( yname ) ? itset.getMetadataY().getName() : yname;
            xunit = isBlank( xunit ) ? itset.getMetadataX().getUnit() : xunit;
            yunit = isBlank( yunit ) ? itset.getMetadataY().getUnit() : yunit;
            final MetadataItem xmeta = new MetadataItem( xname, xunit );
            final MetadataItem ymeta = new MetadataItem( yname, yunit );
            tset = new WrapperTheorySet( itset ) {
                public MetadataItem getMetadataX() {
                    return xmeta;
                }
                public MetadataItem getMetadataY() {
                    return ymeta;
                }
            };
        }
        else {
            tset = itset;
        }
        return new Executable() {
            public void execute() throws IOException {
                TheorySet otset = transformTheorySet( tset, xCex, yCex );
                new ModelTableTheoryWriter()
                    .write( otset, twriter, dest.createStream() );
            }
        };
    }

    /**
     * Transforms a TheorySet.  The output one will be the same as the
     * input one except that the X and Y values may be modified according
     * to given expressions.
     *
     * @param  itset  input theory set
     * @param  xCex   compiled expression for output X vector elements in terms
     *                of X and Y
     * @param  yCex   compiled expression for output Y vector elements in terms
     *                of X and Y
     * @return  transformed theory set
     */
    private static TheorySet transformTheorySet( TheorySet itset,
                                                 CompiledExpression xCex,
                                                 CompiledExpression yCex )
            throws IOException {

        /* If both null output tset will be identical to input. */
        if ( xCex == null && yCex == null ) {
            return itset;
        }

        /* Otherwise transform each constituent theory in turn. */
        int nth = itset.getTheoryCount();
        Theory[] otheories = new Theory[ nth ];
        for ( int ith = 0; ith < nth; ith++ ) {
            otheories[ ith ] =
                transformTheory( itset.getTheory( ith ), xCex, yCex );
        }

        /* Build and return a new theory set based on the transformed
         * theories. */
        return new ArraysTheorySet( otheories,
                                    itset.getMetadataX(),
                                    itset.getMetadataY(),
                                    itset.getMetadataTable() );
    }

    /**
     * Transforms a Theory.  The output one will be the same as the
     * input one except that the X and Y values may be modified according
     * to the given expressions.
     *
     * @param  xCex   compiled expression for output X vector elements in terms
     *                of X and Y
     * @param  yCex   compiled expression for output Y vector elements in terms
     *                of X and Y
     */
    private static Theory transformTheory( Theory th,
                                           CompiledExpression xCex,
                                           CompiledExpression yCex )
            throws IOException {
        TheoryFactory tfact = new TheoryFactory();
        int np = th.getCount();
        Point point = new Point( th );
        Object[] context = new Object[] { point };
        for ( int ip = 0; ip < np; ip++ ) {
            point.index_ = ip;
            double x;
            double y;
            try {
                x = xCex.evaluate_double( context );
                y = yCex.evaluate_double( context );
            }
            catch ( Throwable e ) {
                throw (IOException) 
                      new IOException( "Evaluation error " + e.getMessage() )
                     .initCause( e );
            }
            tfact.addPoint( x, y );
        }
        return tfact.createTheory( th.getName() );
    }

    /**
     * Returns the JEL library to use for expression evaluation.
     * It requires a context containing a single Point object.
     *
     * @return   evaluation library
     */
    private static Library getLibrary() {
        Class[] staticLib =
            (Class[]) JELUtils.getStaticClasses().toArray( new Class[ 0 ] );
        Class[] dynamicLib = new Class[] { Point.class };
        Class[] dotClasses = new Class[ 0 ];
        return new Library( staticLib, dynamicLib, dotClasses, null, null );
    }

    /**
     * Determines whether a string is blank (null or whitespace-only).
     *
     * @param  s  string to test
     * @return   true iff <code>s</code> is blank
     */
    private static boolean isBlank( String s ) {
        return s == null || s.trim().length() == 0;
    }

    /**
     * Helper class used for JEL expression evalaution.
     * Contains public mehods X (or x) and Y (or y) giving theory X and Y
     * coordinates corresponding to the current value of the 
     * <code>index_</code> variable.
     */
    public static final class Point {

        final Theory theory_;

        /** Current index for coordinate evaluation. */
        int index_ = -1;

        /**
         * Constructor.
         *
         * @param   theory  theory to interrogate
         */
        Point( Theory theory ) {
            theory_ = theory;
        }

        /**
         * X coordinate at current index.
         */
        public double x() {
            return theory_.getX( index_ );
        }

        /**
         * Y coordinate at current index.
         */
        public double y() {
            return theory_.getY( index_ );
        }

        /**
         * Synonym for <code>x()</code>.
         */
        public double X() {
            return x();
        }

        /**
         * Synonym for <code>y()</code>.
         */
        public double Y() {
            return y();
        }
    }

    /**
     * Invokes the task.  Try "-help".
     */
    public static void main( String[] args ) {
        TerminalEnvironment.NUM_TRIES = 1;
        new SingleTaskInvoker( new CopyTheorySetTask(), "copymodel" )
           .invoke( args );
    }
}
