package fit.build;

import fit.task.Yafit;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import uk.ac.starlink.task.Parameter;
import uk.ac.starlink.task.SingleTaskInvoker;
import uk.ac.starlink.task.Task;
import uk.ac.starlink.util.LoadException;

/**
 * Used by the documentation build system to write XML source for the
 * description of tasks in the Yafit monolith.
 * Intended to be used from the {@link #main} method.
 *
 * @author   Mark Taylor
 * @since    8 Feb 2007
 */
public class UsageWriter {

    private final String taskName_;
    private final Task task_;

    /**
     * Constructor.
     *
     * @param   taskName  name of the task this writer will describe
     */
    public UsageWriter( String taskName ) throws LoadException {
        taskName_ = taskName;
        task_ = (Task) Yafit.getTaskFactory().createObject( taskName_ );
    }

    /**
     * Outputs a fragment of XML suitable for insertion into the user document
     * which describes this writer's task.
     * The XML is written to standard output.
     */
    public void writeXml() throws IOException {

        /* Write usage. */
        outln( "<p>" );
        outln( "<dl>" );
        outln( "<dt>Usage:</dt>" );
        outln( "<dd><p>" );
        outln( "<verbatim><![CDATA[" );
        new SingleTaskInvoker( task_, taskName_ )
            .invoke( new String[] { "-help" } );
        outln( "]]></verbatim>" );
        outln( "</p></dd>" );

        /* Write parameter descriptions. */
        Parameter[] params = task_.getParameters();
        Arrays.sort(  params, new Comparator() {
            public int compare( Object o1, Object o2 ) {
                Parameter p1 = (Parameter) o1;
                Parameter p2 = (Parameter) o2;
                return ((Parameter) o1).getName()
                      .compareTo( ((Parameter) o2).getName() );
            }
        } );
        outln( "<dt>Parameters:</dt>" );
        outln( "<dd><p>" );
        outln( "<dl>" );
        for ( int ip = 0; ip < params.length; ip++ ) {
            Parameter param = params[ ip ];
            outln( "<dt><code>" + param.getName() + " = <![CDATA[" 
                                + param.getUsage() + "]]>" + "</code></dt>" );
            outln( "<dd>" );
            outln( param.getDescription() );
            String dflt = param.getDefault();
            if ( dflt != null && dflt.length() > 0 ) {
                outln( "<p>[Default: <code><![CDATA[" + dflt
                     + "]]></code>]</p>" );
            }
            outln( "</dd>" );
        }
        outln( "</dl>" );
        outln( "</p></dd>" );
        outln( "</dl>" );
        outln( "</p>" );
    }

    /**
     * Writes a single line to standard output terminated by a newline.
     *
     * @param   line  line to output
     */
    private void outln( String line ) {
        System.out.println( line );
    }

    /**
     * Invokes this writer to describe a given task to standard output.
     * The sole command-line argument should be the name of the task.
     *
     * @param  args  argument vector; 
     */
    public static void main( String[] args ) throws IOException, LoadException {
        if ( args.length != 1 ) {
            System.err.println( "usage: UsageWriter <task-name>" );
            System.exit( 1 );
        }
        new UsageWriter( args[ 0 ] ).writeXml();
    }
}
