package fit.task;

import fit.framework.FitCalculator;
import fit.framework.MetadataItem;
import fit.framework.ObservationSet;
import fit.framework.ResultSink;
import fit.framework.Theory;
import fit.framework.TheorySet;
import fit.run.ResultBestTable;
import fit.run.ResultSummary;
import fit.util.ArraysTheorySet;
import fit.util.ChiSquared;
import fit.util.FitUtils;
import fit.util.InterpolatorFactory;
import fit.util.LeastSquares;
import fit.util.LinearInterpolator;
import fit.util.MagToFluxObservationSet;
import fit.util.Poisson;
import fit.util.SquareSmoother;
import fit.util.TableObservationSetFactory;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.task.BooleanParameter;
import uk.ac.starlink.task.ChoiceParameter;
import uk.ac.starlink.task.Environment;
import uk.ac.starlink.task.Executable;
import uk.ac.starlink.task.OutputStreamParameter;
import uk.ac.starlink.task.Parameter;
import uk.ac.starlink.task.SingleTaskInvoker;
import uk.ac.starlink.task.Task;
import uk.ac.starlink.task.TaskException;
import uk.ac.starlink.task.TerminalEnvironment;
import uk.ac.starlink.ttools.TableConsumer;
import uk.ac.starlink.ttools.task.InputTableParameter;
import uk.ac.starlink.ttools.task.OutputTableParameter;
import uk.ac.starlink.util.Destination;

/**
 * Task which runs the fitting tool.
 *
 * @author   Mark Taylor
 * @since    22 Nov 2006
 */
public class FitTask implements Task {

    private final TheorySetParameter theoryParam_;
    private final ObservationSetParameter obsParam_;
    private final BooleanParameter scaleParam_;
    private final ChoiceParameter calcParam_;
    private final BooleanParameter guiParam_;
    private final OutputStreamParameter summaryOutParam_;
    private final OutputTableParameter bestOutParam_;
    private final OutputTableParameter fullOutParam_;

    private static final String CALC_CHI2 = "chi2";
    private static final String CALC_POISSON = "poisson";
    private static final String CALC_LEASTSQ = "unscaled";

    private static final Logger logger_ = Logger.getLogger( "fit.task" );

    /**
     * Constructor.
     */
    public FitTask() {
        theoryParam_ = new TheorySetParameter( "model" );
        theoryParam_.setDescription( new String[] {
            "<p>File containing the models used for fitting.",
            "This will typically be a <code>ymodel</code> file but may",
            "be in some other format.",
            "The format is given by the",
            "<code>" + theoryParam_.getReaderParameter() + "</code>",
            "parameter.",
            "Multiple model files (in the same format) may be specified",
            "by giving this parameter multiple times.",
            "</p>",
        } );

        obsParam_ = new ObservationSetParameter( "obs" );
        obsParam_.setDescription( new String[] {
            "<p>File containing the observations to be fitted against.",
            "This must be in <code>yobs</code> format.",
            "</p>",
        } );

        scaleParam_ = new BooleanParameter( "scale" );
        scaleParam_.setPrompt( "Scale model curves prior to fit?" );
        scaleParam_.setDescription( new String[] {
            "<p>Whether to scale model curves by a multiplicative constant",
            "to achieve the best fit.",
            "If true, each model curve will be scaled by whatever factor",
            "gives the best fit, otherwise its absolute amplitude will be",
            "used.",
            "</p>",
        } );
        scaleParam_.setDefault( "true" );

        calcParam_ = new ChoiceParameter( "fitcalc", new String[] {
            CALC_CHI2, CALC_POISSON, CALC_LEASTSQ,
        } );
        calcParam_.setPrompt( "Fit scoring algorithm" );
        calcParam_.setDescription( new String[] {
            "<p>Determines how the goodness of fit score will be calculated.",
            "<ul>",
            "<li><code>" + CALC_CHI2 + "</code>: "
                         + "chi squared using value errors</li>",
            "<li><code>" + CALC_POISSON + "</code>: "
                         + "chi squared using Poisson errors</li>",
            "<li><code>" + CALC_LEASTSQ + "</code>: "
                         + "absolute root mean squared distance</li>",
            "</ul>",
            "</p>",
        } );
        calcParam_.setDefault( CALC_CHI2 );

        guiParam_ = new BooleanParameter( "gui" );
        guiParam_.setPrompt( "Display fitting results interactively?" );
        guiParam_.setDescription( new String[] { 
            "<p>If true a window will be displayed plotting the observations",
            "and best fit model curves.",
            "</p>",
        } );
        guiParam_.setDefault( "true" );

        summaryOutParam_ = new OutputStreamParameter( "summary" );
        summaryOutParam_.setNullPermitted( true );
        summaryOutParam_.setDefault( "-" );
        summaryOutParam_.setPrompt( "Output file for results summary" );
        summaryOutParam_.setDescription( new String[] {
            "<p>Gives the destination file for summary output.",
            "This is just a list of the best fit model for each observation.",
            "By default, it is written to standard output (\"-\"),",
            "but it can be sent to a file by giving its name here.",
            "</p>",
        } );

        bestOutParam_ = new OutputTableParameter( "bestfits" );
        bestOutParam_.setNullPermitted( true );
        bestOutParam_.setDefault( null );
        bestOutParam_.setPrompt( "Output table for best fit results" );
        bestOutParam_.getFormatParameter()
                     .setName( bestOutParam_.getName() + "fmt" );
        bestOutParam_.getFormatParameter()
                     .setPrompt( "Table format for "
                               + bestOutParam_.getName() );
        bestOutParam_.getFormatParameter() .setDescription( new String[] {
            "<p>Table format for",
            "<code>" + bestOutParam_.getName() + "</code> table.",
            "Must be one of the permitted STILTS formats such as",
            "<code>votable</code>, <code>fits</code>, <code>ascii</code>,",
            "<code>text</code>, <code>csv</code>, ...",
            "See <docxref doc='sun256' loc='outFormats'/> for a full list.",
            "</p>",
        } );
        bestOutParam_.setDescription( new String[] {
            "<p>Destination filename for a table containing the fitting",
            "results.",
            "This will be a table with one row for each observation",
            "giving the observation details and the details of the model",
            "which fits it best, along with the goodness-of-fit score",
            "and scaling factor if appropriate.",
            "The format of the table is given by the",
            bestOutParam_.getFormatParameter().getName() + " parameter.",
            "</p>",
        } );

        fullOutParam_ = new OutputTableParameter( "allfits" );
        fullOutParam_.setNullPermitted( true );
        fullOutParam_.setDefault( null );
        fullOutParam_.setDescription( "Output table for all fitting results" );
        fullOutParam_.getFormatParameter()
                     .setName( fullOutParam_.getName() + "fmt" );
        fullOutParam_.getFormatParameter()
                     .setPrompt( "Table format for "
                               + fullOutParam_.getName() );
        fullOutParam_.getFormatParameter().setDescription( new String[] {
            "<p>Table format for",
            "<code>" + fullOutParam_.getName() + "</code> table.",
            "</p>",
        } );
        fullOutParam_.setDescription( new String[] {
            "<p>Destination filename for a table containing the complete",
            "fitting results.",
            "This will be a table with a row for each observation-model",
            "fit attempted, so will contain nObs*nModel rows.",
            "Details for each observation and for each model will be",
            "included along with the goodness-of-fit score",
            "and scaling factor if appropriate.",
            "The format of the table is given by the",
            fullOutParam_.getFormatParameter().getName() + " parameter.",
            "</p>",
        } );
    }

    public String getPurpose() {
        return "Perform fitting of observations to model data";
    }

    public Parameter[] getParameters() {
        return new Parameter[] {
            theoryParam_,
            theoryParam_.getReaderParameter(),
            obsParam_,
            obsParam_.getInterpolatorParameter(),
            scaleParam_,
            calcParam_,
            guiParam_,
            summaryOutParam_,
            bestOutParam_,
            bestOutParam_.getFormatParameter(),
            // fullOutParam_,
            // fullOutParam_.getFormatParameter(),
        };
    }

    public Executable createExecutable( Environment env ) throws TaskException {

        /* Get the input theories. */
        final TheorySet theorySet = theoryParam_.theorySetValue( env );

        /* Get the input observations. */
        final ObservationSet obsSet = obsParam_.observationSetValue( env );

        /* Get the fit calculator. */
        final boolean preScale = scaleParam_.booleanValue( env );
        String calcName = (String) calcParam_.objectValue( env );
        final FitCalculator fitcalc;
        if ( CALC_CHI2.equals( calcName ) ) {
            fitcalc = new ChiSquared( preScale ? 1 : 0 );
        }
        else if ( CALC_POISSON.equals( calcName ) ) {
            fitcalc = new Poisson( preScale ? 1 : 0 );
        }
        else if ( CALC_LEASTSQ.equals( calcName ) ) {
            fitcalc = new LeastSquares();
        }
        else {
            throw new AssertionError();
        }

        /* Get the output sinks. */
        List sinkList = new ArrayList();

        /* Prepare for GUI output if requested. */
        if ( guiParam_.booleanValue( env ) ) {
            try {
                sinkList.add( Class.forName( "fit.run.ResultDisplay" )
                                   .newInstance() );
            }
            catch ( Throwable e ) {
                throw new TaskException( "Failed to open GUI", e );
            }
        }

        /* Prepare to write a summary of the results to a stream if
         * requested. */
        Destination summDest = summaryOutParam_.destinationValue( env );
        if ( summDest != null ) {
            sinkList.add( new ResultSummary( summDest ) );
        }

        /* Prepare to write a table representing the best fit theories
         * to a table if requested. */
        final TableConsumer bestConsumer = bestOutParam_.consumerValue( env );
        if ( bestConsumer != null ) {
            sinkList.add( new ResultBestTable() {
                protected void outputTable( StarTable table )
                        throws IOException {
                    bestConsumer.consume( table );
                }
            } );
        }

        /* Prepare to write a table representing all the fits to a table
         * if requested. */
        final TableConsumer fullConsumer = fullOutParam_.consumerValue( env );
        if ( fullConsumer != null ) {
  throw new TaskException( "Full output not yet implemented" );
        }

        /* Construct and validate list of result sinks. */
        final ResultSink[] sinks =
            (ResultSink[]) sinkList.toArray( new ResultSink[ 0 ] );
        if ( sinks.length == 0 ) {
            logger_.warning( "No outputs specified - "
                           + "results will be calculated but discarded" );
        }

        /* Construct and return the object which will do the calculations. */
        return new Executable() {
            public void execute() throws TaskException, IOException {
                doFitting( theorySet, obsSet, preScale, fitcalc, sinks );
            }
        };
    }

    /**
     * Performs the actual fit and dumps the results into the supplied
     * result sinks.
     *
     * @param   theorySet  model data set
     * @param   obsSet    observational data set
     * @param   preScale  whether to prescale the model data to get the best
     *           fit
     * @param   fitCalc   calculator which measures how well curves match
     * @param   sinks    array of result destination handlers
     */
    private static void doFitting( TheorySet theorySet, ObservationSet obsSet,
                                   boolean preScale, FitCalculator fitCalc,
                                   ResultSink[] sinks )
            throws IOException {
        FitUtils.doFitting( theorySet, obsSet, preScale, fitCalc, sinks );
        for ( int is = 0; is < sinks.length; is++ ) {
            sinks[ is ].close();
        }
    }

    /**
     * Invokes the task.  Try "-help".
     */
    public static void main( String[] args ) {
        TerminalEnvironment.NUM_TRIES = 1;
        new SingleTaskInvoker( new FitTask(), "fit" ).invoke( args );
    }
}
