package fit.util;

import fit.framework.Interpolator;
import fit.framework.MetadataItem;
import fit.framework.MetadataTable;
import fit.framework.ObservationSet;

/**
 * ObservationSet implementation based on arrays.
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public class ArraysObservationSet implements ObservationSet {

    private final int npoint_;
    private final int nobs_;
    private final double[] xs_;
    private final double[] xerrs_;
    private final Interpolator[] interps_;
    private final double[][] ys_;
    private final double[][] yerrs_;
    private final double[] xfactors_;
    private final MetadataItem metaX_;
    private final MetadataItem metaY_;
    private final MetadataTable metaTable_;

    /**
     * Constructor.
     *
     * @param   npoint  number of points
     * @param   nobs    number of observation sets
     * @param   xs      npoint-element array of X values
     * @param   xerrs   npoint-element array of X errors
     * @param   interps npoint-element array of interpolators 
     * @param   ys      npoint,nobs-element array of y values
     * @param   yerrs   npoint,nobs-element array of y errors
     * @param   xfactors  nobs-element array of X shift factors, or null
     * @param   metaX   metadata for X values
     * @param   metaY   metadata for Y values
     * @param   metaTable  metadata table, or null
     */
    public ArraysObservationSet( int npoint, int nobs, double[] xs, 
                                 double[] xerrs, Interpolator[] interps,
                                 double[][] ys, double[][] yerrs,
                                 double[] xfactors, MetadataItem metaX,
                                 MetadataItem metaY, MetadataTable metaTable ) {
        if ( xs.length != npoint || xerrs.length != npoint ||
             interps.length != npoint ||
             ys.length != npoint || yerrs.length != npoint ||
             ( xfactors != null && xfactors.length != nobs ) ) {
            throw new IllegalArgumentException( "Array length mismatch" );
        }
        npoint_ = npoint;
        nobs_ = nobs;
        xs_ = xs;
        xerrs_ = xerrs;
        interps_ = interps;
        ys_ = ys;
        yerrs_ = yerrs;
        xfactors_ = xfactors;
        metaX_ = metaX;
        metaY_ = metaY;
        metaTable_ = metaTable;
    }

    public int getObsCount() {
        return nobs_;
    }

    public int getPointCount() {
        return npoint_;
    }

    public double getX( int ip ) {
        return xs_[ ip ];
    }

    public double getXError( int ip ) {
        return xerrs_[ ip ];
    }

    public Interpolator getInterpolator( int ip ) {
        return interps_[ ip ];
    }

    public double getY( int ip, int iobs ) {
        return ys_[ ip ][ iobs ];
    }

    public double getYError( int ip, int iobs ) {
        return yerrs_[ ip ][ iobs ];
    }

    public double getXFactor( int iobs ) {
        return xfactors_ == null ? 1.0 : xfactors_[ iobs ];
    }

    public MetadataItem getMetadataX() {
        return metaX_;
    }

    public MetadataItem getMetadataY() {
        return metaY_;
    }

    public MetadataTable getMetadataTable() {
        return metaTable_;
    }
}
