package fit.util;

import fit.framework.FitCalculator;

/**
 * FitCalculator implementation which uses chi squared fitting.
 * The fitting routines require finite errors for each point.
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public class ChiSquared implements FitCalculator {

    private int nparam_;

    /**
     * Constructor.
     * An integer related to the degrees of freedom is supplied;
     * <code>nparam</code> gives the number of free parameters that are
     * fitted to the data prior to the {@link #getScore} call.
     * This is usually 1 if a prior call to {@link #getScale} will be made,
     * or 0 if not.
     *
     * @param  nparam   number of free parameters fitted to data
     */
    public ChiSquared( int nparam ) {
        nparam_ = nparam;
    }

    public double getScore( int np, double[] y1s, double[] y2s,
                            double[] errs ) {
        double chi2sum = 0.0;
        int nUse = 0;
        for ( int ip = 0; ip < np; ip++ ) {
            double y1 = y1s[ ip ];
            double y2 = y2s[ ip ];
            double err = errs[ ip ];
            double chi = err == 0 ? 0.0
                                  : ( y1 - y2 ) / err;
            if ( ! Double.isNaN( chi ) && ! Double.isInfinite( chi ) ) {
                chi2sum += chi * chi;
                nUse++;
            }
        }
        return nUse > nparam_ ? chi2sum / ( nUse - nparam_ )
                              : Double.NaN;
    }

    public double getScale( int np, double[] y1s, double[] y2s,
                            double[] errs ) {
        double sum12 = 0.0;
        double sum22 = 0.0;
        for ( int ip = 0; ip < np; ip++ ) {
            double y1 = y1s[ ip ];
            double y2 = y2s[ ip ];
            double err = errs[ ip ];
            double err2 = err * err;
            if ( ! Double.isNaN( y1 ) && ! Double.isInfinite( y1 ) &&
                 ! Double.isNaN( y2 ) && ! Double.isInfinite( y2 ) &&
                 err2 > 0.0 && ! Double.isInfinite( err2 ) ) {
                sum12 += ( y1 * y2 ) / err2;
                sum22 += ( y2 * y2 ) / err2;
            }
        }
        return sum12 / sum22;
    }
}
