package fit.util;

import fit.framework.MetadataItem;
import fit.framework.ObservationSet;

/**
 * Transforms an ObservationSet by converting Y values (and associated errors)
 * from AB magnitudes to fluxes in Jansky.
 * The transformation formula is
 * <blockquote>
 *    <i>flux</i> = 10<sup>(23 - (<i>mag</i> + 48.6) / 2.5)
 * </blockquote>
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public class MagToFluxObservationSet extends TransformerObservationSet {

    private final double LOG10 = Math.log( 10.0 );
    private final double K = 1e23 * Math.pow( 10.0, -0.4 * 48.6 );

    /**
     * Constructor.
     *
     * @param   base  observation set which provides its Y values in 
     *          AB magnitudes
     */
    public MagToFluxObservationSet( ObservationSet base ) {
        super( base, MetadataItem.FLUX_JANSKY );
    }

    public double[] transformY( double mag, double magerr ) {
        double flux = K * Math.pow( 10.0, -0.4 * mag );
        double fluxerr = flux * Math.abs( -0.4 ) * LOG10 * magerr;
        return new double[] { flux, fluxerr, };
    }
}
